/*---------------------------------------------------------------------------*\

	FILE....: tec.cpp
	TYPE....: C++ Program
	AUTHOR..: David Rowe
	DATE....: 22/9/01

	Test program to test echo canceler automatically across different
	play and record gains.

	Tips:

	- If you ctrl-c out some of the ports might be left off hook. Run 
	"./offhook" (then hit return straight away) which will take all ports
	off hook, then return then on hook when return is pressed.

	Samples:

	./tec 3 ,,13 -ec (dial 13 out of port 3, echo canceller on)

\*--------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\

         Voicetronix Voice Processing Board (VPB) Software

         Copyright (C) 1999-2001 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
	 USA

\*---------------------------------------------------------------------------*/

#include "../src/mess.h"
#include "../src/comm.h"
#include "../src/config.h"
#include "../src/dspfifo.h"
#include "../src/timer.h"
#include "../src/timer.h"
#include "../src/wobbly.h"
#include "../src/translate.h"
#include "../src/mapdev.h"
#include "../src/vpbapi.h"
extern "C" {
#include "../src/alawmulaw.h"
}

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#define OFF_HK_DELAY    500     // delay after going off hook
#define	N	        160	// size of processing frame
#define NSAM            40      // number of frames to sample
#define NCONV           50      // number of frames to wait for convergence
#define NFREQS          5       // number of test frequencies 
#define FS              8000.0  // sampling frequency
#define AMP             100000.0
#define NR              10
#define NFR             200     // number of frames to test over
#define SINAMP          32767.0

float freqs[] = {300.0, 500.0, 1000.0, 2000.0, 3000.0};

int scan;
FILE *fscan;

void vpb_set_codec_reg(int chdev,    // channel handle
		       unsigned short addr,  // 8-bit address of codec register
		       unsigned short data   // 8-bit data to write to register
		      );

void play_get_hw_gain(int handle, float *gain); 
void play_get_sw_gain(int handle, float *gain); 

void scan_erl(int h, int hdial, float pg, float rg, float swgain, int argc, 
	      char *argv[]);
void scan_erl(int h, int hdial, float pg, float rg, float swgain);
int arg_exists(int argc, char *argv[], char *arg);

// headers for functions internal to driver
int vpb_echo_canc_disable();
int vpb_echo_canc_enable();

void rand_buf(short buf[], int n);
void sin_buf(short buf[], int n);

void coff_get_address(char coff_file[], char symbol[], unsigned long *addr);
extern Comm *vpb_c;			

int kbhit();

int main(int argc, char *argv[])
{
	int       arg,h,hdial;
	int       bal, bal2, bal3, port, on_hook, dport;
	float     playg, recg, swgain;
	VPB_EVENT e;
	char      s[MAX_STR];
	short     thresh;

	// scan command line arguments

	// help
	if (arg_exists(argc, argv, "-h") || arg_exists(argc, argv, "--help")) {
		printf("-bal1 [192..223]   Set codec hybrid balance\n");
		printf("-port [1..4]       VPB port\n");
		printf("-nec               Disable echo canceller\n");
		printf("-pg [-12.0..12.0]  Set codec play gain\n");
		printf("-rg [-12.0..12.0]  Set codec record gain\n");
		printf("-swgain GaindB     Set software gain in dB\n");
		printf("-dial dport number Dial out another port\n");
		printf("-noise             Use noise rather than hello.wav\n");
		printf("-sin               Use sine rather than hello.wav\n");
		exit(0);
	}

	// option to change channel
	if ((arg = arg_exists(argc, argv, "-port")) != 0) {
		port = atoi(argv[arg+1]);
		assert((port >= 1) && (port <= 4));
	}
	else 
		port = 1;
	// initialise 

	try {
		
		h = vpb_open(1,port);

		// option to set codec balance reg 1
		bal = -1;
		if ((arg = arg_exists(argc, argv, "-bal1")) != 0) {
			bal = strtol(argv[arg+1], NULL, 16);
			assert((bal >= 0) && (bal <= 255));
			vpb_set_codec_reg(h, 0x32, bal);
			printf("bal1 = 0x%02x (%d)\n", bal, bal);
		}

		// option to set codec balance reg 2
		bal2 = -1;
		if ((arg = arg_exists(argc, argv, "-bal2")) != 0) {
			bal2 = strtol(argv[arg+1], NULL, 16);
			assert((bal2 >= 0) && (bal2 <= 255));
			vpb_set_codec_reg(h, 0x3a, bal2);
			printf("bal2 = 0x%02x (%d)\n", bal2, bal2);
		}

		// option to set codec balance reg 3
		bal3 = -1;
		if ((arg = arg_exists(argc, argv, "-bal3")) != 0) {
			bal3 = strtol(argv[arg+1], NULL, 16);
			assert((bal3 >= 0) && (bal3 <= 255));
			vpb_set_codec_reg(h, 0x42, bal3);
			printf("bal3 = 0x%02x (%d)\n", bal3, bal3);
		}

		// option to set codec record gain
		if ((arg = arg_exists(argc, argv, "-rg")) != 0) {
			recg = atof(argv[arg+1]);
			assert((recg >= -12.0) && (recg <= 12.0));
			vpb_record_set_hw_gain(h, recg);
		}

		// option to set codec play gain
		if ((arg = arg_exists(argc, argv, "-pg")) != 0) {
			playg = atof(argv[arg+1]);
			assert((playg >= -12.0) && (playg <= 12.0));
			vpb_play_set_hw_gain(h, playg);
		}

		// option to set software (record) gain
		swgain = 1.0;
		if ((arg = arg_exists(argc, argv, "-swgain")) != 0) {
			swgain = atof(argv[arg+1]);
			swgain = pow(10.0,swgain/20.0);
			assert((swgain >= -12.0) && (swgain <= 12.0));
		}

		// option to dissable echo canceller (on by default)
		if (arg_exists(argc, argv, "-nec")) {
			vpb_echo_canc_disable();
		}

		// option to set residual echo suppressor thresh
		if ((arg = arg_exists(argc, argv, "-thresh")) != 0) {
			thresh = atoi(argv[arg+1]);
			vpb_echo_canc_set_sup_thresh(&thresh);
		}			

		// option to dial out of another port
		hdial = -1;
		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
			dport = atoi(argv[arg+1]);
			assert((dport >= 1) && (dport <= 4));
			assert(dport != port);
			hdial = vpb_open(1, dport);
			vpb_sethook_sync(hdial, VPB_OFFHOOK);
			if (arg_exists(argc, argv, "-rg") != 0) {
				vpb_record_set_hw_gain(hdial, recg);
			}
			printf("OK, dialing %s down port %d\n", argv[arg+2],
			       dport);
			vpb_dial_sync(hdial, argv[arg+2]);
		}
		
		// option to enter scan mode 
		arg = arg_exists(argc, argv, "-scan");
		if (arg) {
			scan = atoi(argv[arg+1]);
			assert((scan == 1) || (scan == 2) || (scan == 3));
			fscan = fopen(argv[arg+2],"wt");
			assert(fscan != NULL);

		}
		else
			scan = 0;

		// start main processing loop ---------------------------

		// wait for ring and take off hook

		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
			printf("Waiting for rings on port %d\n",port);
		} else {
			if ((arg = arg_exists(argc, argv, "--offhook")) != 0) {
				vpb_sethook_sync(h,VPB_OFFHOOK);
				on_hook = 0;
			}
			else {
				on_hook = 1;
				printf("Please dial line attached to port"
				       " %d\n",port);
			}
		}

		do {
			while(vpb_get_event_async(&e) == VPB_OK) {
				vpb_translate_event(&e, s);
				mprintf(s);

				// take channels off-hook on ring
				if (e.type == VPB_RING) {
					vpb_sethook_sync(e.handle,VPB_OFFHOOK);
					on_hook = 0;
				}
			}
			vpb_sleep(100);
		} while(on_hook);

		vpb_sleep(OFF_HK_DELAY);

		scan_erl(h, hdial, playg, recg, swgain, argc, argv);

		vpb_sethook_sync(h, VPB_ONHOOK);
		vpb_close(h);
		if ((arg = arg_exists(argc, argv, "-dial")) != 0) {
			vpb_sethook_sync(hdial, VPB_ONHOOK);
			vpb_close(hdial);
		}

	}	// try ...

	catch (Wobbly w) {
		char	s[MAX_STR];
		
		w.translate(s);
		
		if (w.file[0] != 0) 
			printf("exception caught: %s, %s, line = %d\n"
			       ,s, w.file, w.line);
		else
			printf("exception caught: %s\n",s);

		printf("Press any key to exit....\n");
	}

	return 0;
}

// measeure erl using a wave file as input

void scan_erl(int h, int hdial, float pg, float rg, float swgain, int argc, 
	      char *argv[]) {
	float sam;
	int   i,j, arg;
	void  *wave, *wavewr, *waverx, *wavetx;
        short bufrd[N], bufwr[N];
	float echo_pwr, ref_pwr, ref_pwr1, rx_pwr, sum_echo_pwr;
	float erl,gain, converge_frame;

	if (!(arg_exists(argc, argv, "-noise") || arg_exists(argc, argv, "-sin")) ) {
		vpb_wave_open_read(&wave, "hello.wav");
	}
	vpb_wave_open_write(&wavetx, "hellotx.wav", VPB_LINEAR);
	vpb_wave_open_write(&wavewr, "helloec.wav", VPB_LINEAR);
	vpb_wave_open_write(&waverx, "hellorx.wav", VPB_LINEAR);

	vpb_record_buf_start(h, VPB_LINEAR);
	vpb_play_buf_start(h, VPB_LINEAR);
	if (hdial != -1) {
		vpb_record_buf_start(hdial,VPB_LINEAR);
		if ((arg=arg_exists(argc, argv, "-ne"))) {
			// simulate near end speech
			printf("file: %s\n",argv[arg+1]);
			vpb_play_file_async(hdial, argv[arg+1], 0);
		}
	}
	
	// prime play buffer
	for(i=0; i<10; i++) {
		if (arg_exists(argc, argv, "-noise")) {
		        rand_buf(bufwr,N);
		} 
		else {
		  if (arg_exists(argc, argv, "-sin")) {
		    sin_buf(bufwr,N);
		  }
		  else {
		    vpb_wave_read(wave,(char*)bufwr,sizeof(short)*N);
		  }
		}
		vpb_play_buf_sync(h, (char*)bufwr, sizeof(short)*N);
	}

	// prime record buffer
	for(i=0; i<2; i++) {
		vpb_record_buf_sync(h, (char*)bufrd, sizeof(short)*N);
	}

	sum_echo_pwr = ref_pwr = rx_pwr = 0.0;
	converge_frame = -1;
	
	for(i=0; i<NFR; i++) {
	  if (arg_exists(argc, argv, "-noise")) {
	    rand_buf(bufwr,N);
	  }
	  else {
	    if (arg_exists(argc, argv, "-sin")) {
	      sin_buf(bufwr,N);
	    }
	    else {
	      vpb_wave_read(wave,(char*)bufwr,sizeof(short)*N);
	    }
	  }
	  // ensure file saved to disk has same alaw processing as
	  // file played
	  char charbuf[N];
		alaw_encode(charbuf, bufwr, N);
		alaw_decode(bufwr, charbuf, N);
		vpb_wave_write(wavetx,(char*)bufwr, sizeof(short)*N);

		vpb_play_buf_sync(h, (char*)bufwr, sizeof(short)*N);
		ref_pwr1 = 0.0;
		for(j=0; j<N; j++) { 
			sam = bufwr[j];
			if (i > NFR/4) ref_pwr += sam*sam;
		}

		vpb_record_buf_sync(h, (char*)bufrd, sizeof(short)*N);
		for(j=0; j<N; j++) {
			sam = swgain*bufrd[j];
			bufrd[j] = (short)sam;
		}
		vpb_wave_write(wavewr,(char*)bufrd, sizeof(short)*N);
		echo_pwr = 0.0;
		for(j=0; j<N; j++) { 
			sam = bufrd[j];
			echo_pwr += sam*sam;
		} 
		if (i > NFR/4) sum_echo_pwr += echo_pwr;
		if (echo_pwr > 0)
			converge_frame = i;
		if (hdial != -1) {
			vpb_record_buf_sync(hdial, (char*)bufrd, 
					    sizeof(short)*N);
			for(j=0; j<N; j++) {
				sam = swgain*bufrd[j];
				bufrd[j] = (short)sam;
			}
			vpb_wave_write(waverx,(char*)bufrd,sizeof(short)*N);
			for(j=0; j<N; j++) { 
				sam = bufrd[j];
				if (i > NFR/4) rx_pwr += sam*sam;
			} 

		}
	}

	vpb_play_buf_finish(h);
	vpb_record_buf_finish(h);
	if (hdial != -1) vpb_record_buf_finish(hdial);
	if (!(arg_exists(argc, argv, "-noise") || arg_exists(argc, argv, "-sin"))) {
		vpb_wave_close_read(wave);
	}
	vpb_wave_close_write(wavetx);
	vpb_wave_close_write(wavewr);
	if (hdial != -1) {
		vpb_wave_close_write(waverx);
	}

	erl = -10*log10(ref_pwr/sum_echo_pwr);
	gain = 10*log10(rx_pwr/ref_pwr);
	printf("ERL: %5.2f  GAIN: %5.2f\n",erl,gain);
	if (!arg_exists(argc, argv, "-nec")) {
		printf("Converged in %3.2f ms\n", converge_frame*N*1000.0/FS);
		printf("Converged in %f frames\n", converge_frame);
	}
}

int arg_exists(int argc, char *argv[], char *arg) {
  int i;

  for(i=0; i<argc; i++)
    if (strcmp(argv[i],arg) == 0)
      return i;

  return 0;
}

// generates a buffer of gaussian random samples
void rand_buf(short buf[], int n) {
	int i,j;
	float sam;

	for(i=0; i<n; i++) {
		sam = 0.0;
		for(j=0; j<NR; j++) {
			sam += (float)rand()/RAND_MAX - 0.5;
		}
		sam *= AMP/NR;
		/*
		if (sam >  8192) sam = 8192;
		if (sam < -8192) sam = -8192;
		*/
		buf[i] = (short)sam;
	}

}

// generates a buffer of sinusoidal samples
void sin_buf(short buf[], int n) {
	int        i;
	static int t;
	float      sam;

	for(i=0; i<n; i++) {
		sam = SINAMP*cos(2.0*3.141*(t++)*1000.0/8000.0);
		buf[i] = (short)sam;

	}
}




